%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%
%%  STANDARD CELL %%
%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%

% Standard base cell with user-tunable parameters: below are the
% layers composing the cell

% 1) SiO2
% 2) Titanium
% 3) Gold (bottom)
% 4) Vacuum + HfO2
% 5) Gold (top) + Titanium side film + SiO2 filler

clear all; close all; clc;

%% ===================== USER PARAMETERS ===============================
% Base (SiO2)
L = 24;     % along X [nm]
W = 10;     % along Y [nm]
H = 2;      % along Z [nm]
origin = [0 0 0];   % [x0 y0 z0] in nm

% Bottom Titanium layer (only height is variable)
H_Ti_bottom   = 1;  % [nm]

% Bottom Gold layer (only height is variable)
H_gold_bottom = 2;  % [nm]

% Vacuum (variable X length and Z height; Y = W)
L_vac = 4;          % length along X [nm]
H_vac = 3;          % height along Z [nm]

% Top Gold cap (all dimensions are variable)
L_Au_top = 16;      % X [nm]
W_Au_top = 8;       % Y [nm]
H_Au_top = 4;       % Z [nm]

% Lateral film around the vertical faces of the top Au (material: Ti)
T_film   = 0.5;     % thickness [nm] (variable)
% ======================================================================

% Plot style
faceColor_base  = [0.55 0.35 0.20]; % brown (SiO2)
faceColor_Ti    = [0.20 0.20 0.20]; % dark gray (Ti)
faceColor_Au    = [1.00 0.84 0.00]; % yellow (Au)  <-- also used for top cap
faceColor_vac   = [0.00 0.60 0.00]; % green (vacuum) transparent
alpha_vac       = 0.35;
faceColor_hfo2  = [0.80 0.65 0.45]; % light brown (HfO2) opaque
edgeColor       = 'k';
lineWidth       = 1.0;

% Labels
labelBaseText   = 'SiO2';
labelTiText     = 'Ti';
labelAuText     = 'Au';
labelVacText    = 'vacuum';
labelHfO2Text   = 'HfO2';
labelBaseColor  = [0 0 0];
labelTiColor    = [1 1 1];
labelAuColor    = [0 0 0];
labelVacColor   = [0 0.2 0];
labelHfO2Color  = [0 0 0];
labelSize       = 12;
% ======================================================================

% ------------------ LOCAL FUNCTION FOR VERTICES & FACES ----------------
vertices_box = @(o,Lx,Wy,Hz) [ o;                     % p0
                               o + [Lx 0 0];          % p1
                               o + [0 Wy 0];          % p2
                               o + [0 0 Hz];          % p3
                               o + [Lx Wy 0];         % p4
                               o + [Lx 0 Hz];         % p5
                               o + [0 Wy Hz];         % p6
                               o + [Lx Wy Hz] ];      % p7

F = [ 1 2 5 3;   % bottom
      3 5 8 7;   % side Y+
      1 3 7 4;   % side X-
      2 6 8 5;   % side X+
      1 4 6 2;   % side Y-
      4 7 8 6 ]; % top

% --------------------------- BASE BOX ----------------------------------
o = origin(:).';
V_base = vertices_box(o, L, W, H);

% -------------------------- UPPER BOX (Ti) -----------------------------
o_Ti = o + [0 0 H];
V_Ti = vertices_box(o_Ti, L, W, H_Ti_bottom);

% -------------------------- UPPER BOX (Au_bottom) ----------------------
o_Au = o + [0 0 (H + H_Ti_bottom)];
V_Au = vertices_box(o_Au, L, W, H_gold_bottom);

% -------------- UPPER LEVEL: HfO2 — vacuum — HfO2 ---------------------
z0_top = o(3) + H + H_Ti_bottom + H_gold_bottom;

% Clamp L_vac in [0, L] and compute lateral HfO2 widths
L_vac_eff = min(max(L_vac, 0), L);
L_left  = max(0, (L - L_vac_eff)/2);
L_right = max(0, L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vertices_box(o_left,  L_left,    W, H_vac); % left HfO2
V_vac   = vertices_box(o_vac,   L_vac_eff, W, H_vac); % vacuum
V_right = vertices_box(o_right, L_right,   W, H_vac); % right HfO2

% ----------------------- TOP Au CAP ------------------------------------
x0_cap = o(1) + (L - L_Au_top)/2;
y0_cap = o(2) + (W - W_Au_top)/2;
z0_cap = z0_top + H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vertices_box(o_cap, L_Au_top, W_Au_top, H_Au_top);

% ------------------------ LATERAL FILM (Ti) ----------------------------
% 4 vertical walls around the top Au (Titanium color)
o_film_xL = [x0_cap - T_film,      y0_cap,            z0_cap];     % X- side
o_film_xR = [x0_cap + L_Au_top,    y0_cap,            z0_cap];     % X+ side
V_film_xL = vertices_box(o_film_xL, T_film,           W_Au_top, H_Au_top);
V_film_xR = vertices_box(o_film_xR, T_film,           W_Au_top, H_Au_top);

o_film_yF = [x0_cap - T_film,      y0_cap - T_film,   z0_cap];     % Y- side
o_film_yB = [x0_cap - T_film,      y0_cap + W_Au_top, z0_cap];     % Y+ side
V_film_yF = vertices_box(o_film_yF, L_Au_top + 2*T_film, T_film,   H_Au_top);
V_film_yB = vertices_box(o_film_yB, L_Au_top + 2*T_film, T_film,   H_Au_top);

% -------------------- SiO2 FRAME (same color as base) ------------------
% inner hole of the frame = outer size of the Ti film
x_in_min = x0_cap - T_film;
x_in_max = x0_cap + L_Au_top + T_film;
y_in_min = y0_cap - T_film;
y_in_max = y0_cap + W_Au_top + T_film;

% outer footprint of the frame = entire SiO2 base footprint
x_base_min = o(1);       x_base_max = o(1) + L;
y_base_min = o(2);       y_base_max = o(2) + W;

% Frame thicknesses (>=0)
T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0, y_base_max - y_in_max);

% 4 non-overlapping walls:
%    - Left / Right span the whole Y of the footprint
%    - Front / Back only span from x_in_min to x_in_max (to avoid overlap with L/R)
o_frame_xL = [x_base_min, y_base_min, z0_cap];
o_frame_xR = [x_in_max,   y_base_min, z0_cap];
V_frame_xL = vertices_box(o_frame_xL, T_left_frame,  W,         H_Au_top);
V_frame_xR = vertices_box(o_frame_xR, T_right_frame, W,         H_Au_top);

o_frame_yF = [x_in_min,   y_base_min, z0_cap];
o_frame_yB = [x_in_min,   y_in_max,   z0_cap];
V_frame_yF = vertices_box(o_frame_yF, x_in_max - x_in_min, T_front_frame, H_Au_top);
V_frame_yB = vertices_box(o_frame_yB, x_in_max - x_in_min, T_back_frame,  H_Au_top);

% ------------------------------- PLOT ----------------------------------
figure('Color','w'); hold on;

% Draw with DisplayName to control legend entries
p1 = patch('Vertices',V_base,'Faces',F, ...
      'FaceColor',faceColor_base, 'EdgeColor',edgeColor, ...
      'LineWidth',lineWidth, 'DisplayName','SiO2');

p2 = patch('Vertices',V_Ti,'Faces',F, ...
      'FaceColor',faceColor_Ti, 'EdgeColor',edgeColor, ...
      'LineWidth',lineWidth, 'DisplayName','Ti');

p3 = patch('Vertices',V_Au,'Faces',F, ...
      'FaceColor',faceColor_Au, 'EdgeColor',edgeColor, ...
      'LineWidth',lineWidth, 'DisplayName','Au');   % bottom

% Left HfO2
p4 = [];
if L_left > 0
    p4 = patch('Vertices',V_left,'Faces',F, ...
        'FaceColor',faceColor_hfo2, 'EdgeColor',edgeColor, ...
        'LineWidth',lineWidth, 'DisplayName','HfO2');
end

% Vacuum (legend entry)
p5 = [];
if L_vac_eff > 0
    p5 = patch('Vertices',V_vac,'Faces',F, ...
        'FaceColor',faceColor_vac, 'FaceAlpha',alpha_vac, ...
        'EdgeColor',edgeColor, 'LineWidth',lineWidth, ...
        'DisplayName','vacuum');
end

% Right HfO2
if L_right > 0
    patch('Vertices',V_right,'Faces',F, ...
        'FaceColor',faceColor_hfo2, 'EdgeColor',edgeColor, ...
        'LineWidth',lineWidth, 'HandleVisibility','off');
end

% Top Au cap
patch('Vertices',V_cap,'Faces',F, ...
      'FaceColor',faceColor_Au, 'EdgeColor',edgeColor, ...
      'LineWidth',lineWidth, 'HandleVisibility','off');

% Lateral film (Ti) 
patch('Vertices',V_film_xL,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch('Vertices',V_film_xR,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch('Vertices',V_film_yF,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');
patch('Vertices',V_film_yB,'Faces',F,'FaceColor',faceColor_Ti,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off');

% SiO2 frame (same color as base) 
if T_left_frame  > 0, patch('Vertices',V_frame_xL,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if T_right_frame > 0, patch('Vertices',V_frame_xR,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if T_front_frame > 0, patch('Vertices',V_frame_yF,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end
if T_back_frame  > 0, patch('Vertices',V_frame_yB,'Faces',F,'FaceColor',faceColor_base,'EdgeColor',edgeColor,'LineWidth',lineWidth,'HandleVisibility','off'); end

% Axes and view
axis equal; grid on; view(35,25);
xlabel('X [nm]'); ylabel('Y [nm]'); zlabel('Z [nm]');
% title(sprintf(['Stack: SiO2 (%.1f) + Ti (%.1f) + Au (%.1f) + ', ...
%                '[HfO2 | vac | HfO2] (H=%.1f) + Au_top (%.1f×%.1f×%.1f) + ', ...
%                'Film_Ti(T=%.1f) + SiO2 frame (H=%.1f) [nm]'], ...
%                H, H_Ti_bottom, H_gold_bottom, H_vac, L_Au_top, W_Au_top, H_Au_top, ...
%                T_film, H_Au_top));
title('Standard Cell');

% Axis limits with margins
H_tot = H + H_Ti_bottom + H_gold_bottom + H_vac + H_Au_top;
marg = 0.1 * [L W H_tot];
xlim([o(1)-marg(1), o(1)+L+marg(1)]);
ylim([o(2)-marg(2), o(2)+W+marg(2)]);
zlim([o(3)-marg(3), o(3)+H_tot+marg(3)]);

% Guide axes 
plot3(o(1),o(2),o(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
quiver3(o(1),o(2),o(3), L,0,0, 0, 'LineWidth',1, 'HandleVisibility','off');
quiver3(o(1),o(2),o(3), 0,W,0, 0, 'LineWidth',1, 'HandleVisibility','off');
quiver3(o(1),o(2),o(3), 0,0,H_tot, 0, 'LineWidth',1, 'HandleVisibility','off');

% ------------------------------- LABELS --------------------------------
% Base layer
C_base = o + [L W H]/2;
text(C_base(1), C_base(2), C_base(3), labelBaseText, ...
     'HorizontalAlignment','center','VerticalAlignment','middle', ...
     'FontWeight','bold','FontSize',labelSize,'Color',labelBaseColor);

C_Ti = o_Ti + [L W H_Ti_bottom]/2;
text(C_Ti(1), C_Ti(2), C_Ti(3), labelTiText, ...
     'HorizontalAlignment','center','VerticalAlignment','middle', ...
     'FontWeight','bold','FontSize',labelSize,'Color',labelTiColor);

C_Au = o_Au + [L W H_gold_bottom]/2;
text(C_Au(1), C_Au(2), C_Au(3), labelAuText, ...
     'HorizontalAlignment','center','VerticalAlignment','middle', ...
     'FontWeight','bold','FontSize',labelSize,'Color',labelAuColor);

% Single HfO2 label
if L_left > 0
    C_HfO2 = o_left + [L_left W H_vac]/2;
elseif L_right > 0
    C_HfO2 = o_right + [L_right W H_vac]/2;
else
    C_HfO2 = [];
end
if ~isempty(C_HfO2)
    text(C_HfO2(1), C_HfO2(2), C_HfO2(3), labelHfO2Text, ...
         'HorizontalAlignment','center','VerticalAlignment','middle', ...
         'FontWeight','bold','FontSize',labelSize,'Color',labelHfO2Color);
end

% Vacuum label
if L_vac_eff > 0
    C_vac = o_vac + [L_vac_eff W H_vac]/2;
    text(C_vac(1), C_vac(2), C_vac(3), labelVacText, ...
         'HorizontalAlignment','center','VerticalAlignment','middle', ...
         'FontWeight','bold','FontSize',labelSize,'Color',labelVacColor);
end

% ------------------------------- LEGEND --------------------------------
lh = [];  % collect only plotted objects to be shown in the legend
if isgraphics(p1), lh(end+1) = p1; end
if isgraphics(p2), lh(end+1) = p2; end
if isgraphics(p3), lh(end+1) = p3; end
if isgraphics(p4), lh(end+1) = p4; end
if isgraphics(p5), lh(end+1) = p5; end
legend(lh, 'Location','bestoutside');
